/* hardcoded_angles_from_file.cpp
 * Aligns point clouds that have been turned 22.5 degrees each time, a total of 16 clouds, and level with the camera.
 * They must be turned 1 meter away from the camera.
 * Created by 3d-scanning on the cheap
 */

#include <stdio.h>
#include <iostream>
#include <queue>
#include <pcl/io/openni_grabber.h>
#include <pcl/io/pcd_io.h>
#include <pcl/io/file_io.h>
#include <pcl/io/obj_io.h>
#include <pcl/visualization/cloud_viewer.h>
#include <pcl/point_types.h>
#include <pcl/filters/filter.h>
#include <pcl/filters/passthrough.h>
#include <pcl/common/transforms.h>

class Comps_HC_Angles
{
public:
	void run ()
	{
		printf("Starting... \n");
		// Get files from computer and put them in the queue
		loadFiles(16);
		doRotation();
		printf("ending run()\n");
	}

private:
	std::queue<pcl::PointCloud<pcl::PointXYZRGBA>::Ptr > cloud_queue_;

	//Loads .pcd files and returns them in a queue. numClouds is the number of clouds to load, inclusive
	void loadFiles(int numClouds){
		for(int i=0;i<numClouds;i++){
			pcl::PointCloud<pcl::PointXYZRGBA>::Ptr curCloud (new pcl::PointCloud<pcl::PointXYZRGBA>);
			std::string filename ("inputCloud" + std::to_string(static_cast<long long>(i)) + ".pcd");
			if(pcl::io::loadPCDFile<pcl::PointXYZRGBA> (filename, *curCloud) == -1){
				printf("Couldnt load file: %s\n", filename);
				return;
			}			
			Eigen::Affine3f transformation = Eigen::Affine3f::Identity();

			// Moving point cloud 1 meter along z axis, to make place it at the center 
			// (inspired by this tutorial: http://www.pointclouds.org/documentation/tutorials/matrix_transform.php#matrix-transform)
			transformation.translation() << 0.0, 0.0, -1.0;
			pcl::transformPointCloud(*curCloud, *curCloud, transformation);
			// Remove noise around boot
			curCloud = filterBounds(curCloud);
			cloud_queue_.push(curCloud);
		}
	}

	// Applies the correct rotation matrix to each point cloud, so that the boot is correctly aligned
	void doRotation(){
		printf("this is ROTATION \n");

		pcl::PointCloud<pcl::PointXYZRGBA>::Ptr sourceCloudPtr (new pcl::PointCloud<pcl::PointXYZRGBA>);
		pcl::PointCloud<pcl::PointXYZRGBA>::Ptr finalCloudPtr (new pcl::PointCloud<pcl::PointXYZRGBA>); // The collection of all of the point clouds correctly aligned

		// Go through queue and rotate cloud by 22.5 degrees, and put it all on the finalCloudPtr
		while(cloud_queue_.size() > 1){

			sourceCloudPtr = cloud_queue_.front();
			cloud_queue_.pop();

			std::vector<int> indices;
			pcl::removeNaNFromPointCloud(*sourceCloudPtr, *sourceCloudPtr, indices);

			pcl::PointCloud<pcl::PointXYZRGBA>::Ptr transformedCloudPtr (new pcl::PointCloud<pcl::PointXYZRGBA>); //Declaring new cloud here just to be safe

			// Rotating sourceCloudPtr around y axis by 22.5 degrees (inspired by this tutorial: http://www.pointclouds.org/documentation/tutorials/matrix_transform.php#matrix-transform)
			float theta = (16 - cloud_queue_.size()) * 22.5 * (M_PI/180.0);  // Converting degrees to radians, and doing a cumulative rotation
			Eigen::Affine3f rotation = Eigen::Affine3f::Identity();
			rotation.rotate (Eigen::AngleAxisf (theta, Eigen::Vector3f::UnitY()));
			pcl::transformPointCloud(*sourceCloudPtr, *transformedCloudPtr, rotation);
			*finalCloudPtr += *transformedCloudPtr; //Add transformed cloud to final model
			printf("Cloud queue size in loop: %i \n", cloud_queue_.size());
		}
		// save as .pcd
		pcl::io::savePCDFile("frames_" + std::to_string(static_cast<long long>(16)) + "_3dmodel.pcd", *finalCloudPtr);
		view(finalCloudPtr);
	}

	//Removes points in cloud outside of a cube of length size (in meters)
	//Assumes object is centered 1m away from camera, and that it is Jonathon's boot
	pcl::PointCloud<pcl::PointXYZRGBA>::Ptr filterBounds(pcl::PointCloud<pcl::PointXYZRGBA>::Ptr cloud){
		//view(cloud);
		//Initialize filter and cloud to return
		pcl::PassThrough<pcl::PointXYZRGBA> pass;
		pass.setInputCloud(cloud);
		//We need a new cloud instead of modifying in place
		// because of a PCL bug that only affects windows
		pcl::PointCloud<pcl::PointXYZRGBA>::Ptr filtered1 (new pcl::PointCloud<pcl::PointXYZRGBA>);
		pcl::PointCloud<pcl::PointXYZRGBA>::Ptr filtered2 (new pcl::PointCloud<pcl::PointXYZRGBA>);

		//Filter x coordinate
		pass.setFilterFieldName("x");
		pass.setFilterLimits(-0.15, 0.15);
		pass.filter(*filtered1);

		//Filter y coordinate
		pass.setInputCloud(filtered1);
		pass.setFilterFieldName("y");
		pass.setFilterLimits(-0.5,0);
		pass.filter(*filtered2);
				
		//Filter z coordinate
		pass.setInputCloud(filtered2);
		pass.setFilterFieldName("z");
		pass.setFilterLimits(-0.4, 0.4);
		pass.filter(*filtered1);
		return filtered1;
	}

	// Shows the passed in cloud in a viewer
	void view(pcl::PointCloud<pcl::PointXYZRGBA>::Ptr cloud){
		boost::shared_ptr<pcl::visualization::CloudViewer> viewer (new pcl::visualization::CloudViewer ("Jonathon's Boot!"));
		viewer->showCloud(cloud);
		cin.get();
	}	
};

int main (int argc)
{
	Comps_HC_Angles hardcodedAngles;
	hardcodedAngles.run ();
	return 0;
}